import { getDictionary, Locale } from "@/lib/i18n";
import { ContactForm } from "@/components/sections/ContactForm";
import { Card, CardContent } from "@/components/ui/card";
import { Mail, MapPin, Phone } from "lucide-react";

export default async function ContactPage({
  params,
}: {
  params: { lang: Locale };
}) {
  const { lang } = params;
  const dictionary = await getDictionary(lang);
  const { contact } = dictionary;

  return (
    <div className="flex flex-col min-h-screen">
      {/* Hero Section */}
      <section className="py-20 md:py-32 bg-gradient-to-br from-brand-primary/5 via-white to-brand-secondary/5">
        <div className="container px-4 md:px-6">
          <div className="max-w-3xl mx-auto text-center space-y-6">
            <h1 className="text-4xl md:text-6xl font-bold tracking-tighter bg-clip-text text-transparent bg-gradient-to-r from-brand-primary to-brand-secondary">
              {contact.hero.title}
            </h1>
            <p className="text-xl text-text-secondary max-w-2xl mx-auto">
              {contact.hero.subtitle}
            </p>
          </div>
        </div>
      </section>

      {/* Contact Section */}
      <section className="py-20 md:py-24">
        <div className="container px-4 md:px-6">
          <div className="max-w-6xl mx-auto grid md:grid-cols-2 gap-12">
            {/* Contact Form */}
            <ContactForm lang={lang} dict={contact.form} />

            {/* Contact Info */}
            <div className="space-y-6">
              <h2 className="text-2xl md:text-3xl font-bold mb-6">
                {contact.info.title}
              </h2>

              <Card>
                <CardContent className="p-6 space-y-6">
                  <div className="flex items-start gap-4">
                    <div className="w-12 h-12 rounded-lg bg-brand-primary/10 flex items-center justify-center flex-shrink-0">
                      <MapPin className="w-6 h-6 text-brand-primary" />
                    </div>
                    <div>
                      <h3 className="font-semibold mb-1">
                        {lang === "en" ? "Address" : lang === "ru" ? "Адрес" : "Manzil"}
                      </h3>
                      <p className="text-text-secondary">
                        {contact.info.address}
                      </p>
                    </div>
                  </div>

                  <div className="flex items-start gap-4">
                    <div className="w-12 h-12 rounded-lg bg-brand-secondary/10 flex items-center justify-center flex-shrink-0">
                      <Phone className="w-6 h-6 text-brand-secondary" />
                    </div>
                    <div>
                      <h3 className="font-semibold mb-1">
                        {lang === "en" ? "Phone" : lang === "ru" ? "Телефон" : "Telefon"}
                      </h3>
                      <p className="text-text-secondary">{contact.info.phone}</p>
                    </div>
                  </div>

                  <div className="flex items-start gap-4">
                    <div className="w-12 h-12 rounded-lg bg-accent-purple/10 flex items-center justify-center flex-shrink-0">
                      <Mail className="w-6 h-6 text-accent-purple" />
                    </div>
                    <div>
                      <h3 className="font-semibold mb-1">Email</h3>
                      <p className="text-text-secondary">{contact.info.email}</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="bg-gradient-to-br from-brand-primary to-brand-secondary text-white">
                <CardContent className="p-6">
                  <h3 className="text-xl font-semibold mb-2">
                    {lang === "en" ? "Working Hours" : lang === "ru" ? "Часы работы" : "Ish vaqti"}
                  </h3>
                  <p className="opacity-90">
                    {lang === "en" 
                      ? "Monday - Friday: 9:00 - 18:00"
                      : lang === "ru"
                      ? "Понедельник - Пятница: 9:00 - 18:00"
                      : "Dushanba - Juma: 9:00 - 18:00"}
                  </p>
                  <p className="opacity-90">
                    {lang === "en" ? "Saturday - Sunday: Closed" : lang === "ru" ? "Суббота - Воскресенье: Выходной" : "Shanba - Yakshanba: Dam olish"}
                  </p>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
}
